function rungeKutta(obj, stepSize)
%RUNGEKUTTA Simulate a StateSpaceModel using the classical Runge-Kutta method RK4
% All other details are the same as in runSim

% David Katzin, Wageningen University
% david.katzin@wur.nl

	tStart = obj.t.val(1);
	tEnd = obj.t.val(2);
	timePhase = tStart:stepSize:(tEnd-stepSize);
    [stateNames, auxNames, ctrlNames, paramNames, inputNames] = getFieldNames(obj);
    
    if strcmp(obj.u.(ctrlNames{1}).def, ctrlNames{1}) 
        % the def for the first control is the control name
        ruleBased = false;
    else
        ruleBased = true;
    end
    
    [x, a, u] = createBlankTrajectories(obj, length(timePhase));
    
    %% load inital values
    [x0, u0, a0, p, d0]  = getInitialValues(obj);

    % place initial values in trajectories
    x = insertValue(x,x0,1);
    a = insertValue(a,a0,1);   
    u = insertValue(u,u0,1);
    
    pb = CmdLineProgressBar('Progress: ',datetime('now')); % progress bar
    
    %% Calculate trajectories
    for n = 2:length(timePhase)
        pb.print(100*n/length(timePhase),100);
        
        t = timePhase(n);    
        tHalf = 1/2*(timePhase(n)+timePhase(n-1));
        
        % get dHalf = d(t-h/2)
        for k=1:length(inputNames)
            if t < obj.d.(inputNames{k}).val(1,1)
                dHalf.(inputNames{k}) = obj.d.(inputNames{k}).val(1,2);
            elseif t > obj.d.(inputNames{k}).val(end,1)
                dHalf.(inputNames{k}) = obj.d.(inputNames{k}).val(end,2);
            else
                dHalf.(inputNames{k}) = interp1(obj.d.(inputNames{k}).val(:,1),...
                    obj.d.(inputNames{k}).val(:,2),tHalf); 
            end
        end
        
        % get d1 = d(t)
        for k=1:length(inputNames)
            if t < obj.d.(inputNames{k}).val(1,1)
                d1.(inputNames{k}) = obj.d.(inputNames{k}).val(1,2);
            elseif t > obj.d.(inputNames{k}).val(end,1)
                d1.(inputNames{k}) = obj.d.(inputNames{k}).val(end,2);
            else
                d1.(inputNames{k}) = interp1(obj.d.(inputNames{k}).val(:,1),...
                    obj.d.(inputNames{k}).val(:,2),t); 
            end
        end
        
        % get uHalf = u(t-h/2)
        if ruleBased % get from rules
            uHalf = getControls(obj, x0, a0, u0, d0, p);
        else % get from trajectory
            for k=1:length(ctrlNames)
                uHalf.(ctrlNames{k}) = interp1(obj.u.(ctrlNames{k}).val(:,1),...
                    obj.u.(ctrlNames{k}).val(:,2),tHalf); 
            end
        end
        
        % get u1 = u(t)
        if ruleBased % get from rules
            u1 = getControls(obj, x0, a0, u0, d0, p);
        else % get from trajectory
            for k=1:length(ctrlNames)
                u1.(ctrlNames{k}) = interp1(obj.u.(ctrlNames{k}).val(:,1),...
                    obj.u.(ctrlNames{k}).val(:,2),t); 
            end
        end
        
        %% Calculate x1 = x(t), a1 = a(t)
        [x1, a1]  = getNextStep(obj, stepSize, x0, a0, u0, uHalf, u1, d0, dHalf, d1, p);
        
        % store new values
        x = insertValue(x,x1,n);
        a = insertValue(a,a1,n);
        u = insertValue(u,u1,n);
        
        % prepare next round
        x0 = x1;
        u0 = u1;
        a0 = a1;
        d0 = d1;
        
       
    end

    if ~ruleBased
        u = []; % can throw this away
    end
    copyValues(obj, x, a, u, timePhase); % if isempty(u), u will not be copied
    
end


function copyValues(obj, x, a, u, timePhase)
    % Copy the calculated trajectories x,a,u to their appropriate place in obj
    [stateNames, auxNames, ctrlNames, ~, ~] = getFieldNames(obj);
    
     for n=1:length(stateNames)
        obj.x.(stateNames{n}).val = [];
        obj.x.(stateNames{n}).val(:,1) = timePhase;
        obj.x.(stateNames{n}).val(:,2) = x.(stateNames{n});
    end
    
    for n=1:length(auxNames)
        obj.a.(auxNames{n}).val = [];
        obj.a.(auxNames{n}).val(:,1) = timePhase;
        obj.a.(auxNames{n}).val(:,2) = a.(auxNames{n});
    end
    
    if ~isempty(u)
        for n=1:length(ctrlNames)
            obj.u.(ctrlNames{n}).val = [];
            obj.u.(ctrlNames{n}).val(:,1) = timePhase;
           obj.u.(ctrlNames{n}).val(:,2) = u.(ctrlNames{n});
        end
    end
end

function [x, a, u] = createBlankTrajectories(obj, trajSize)
    [stateNames, auxNames, ctrlNames, ~, ~] = getFieldNames(obj);
    
    for n=1:length(stateNames)
        x.(stateNames{n}) = NaN(trajSize,1);
    end

    for n=1:length(auxNames)
        a.(auxNames{n}) = NaN(trajSize,1);
    end
    
    for n=1:length(ctrlNames)
        u.(ctrlNames{n}) = NaN(trajSize,1);
    end
end

function z = insertValue(z,z0,n)
    % z - a struct with fields as vectors, all at the same length
    % z0 - a struct with the same fields, but all scalar values
    % for every field in z, the following will happen:
    %   z.<field>(n) = z0.field

    names = fieldnames(z0);
    
    for k=1:length(names)
        z.(names{k})(n) = z0.(names{k});
    end
end

function [x, u, a, p, d]  = getInitialValues(obj)

    % by default return empty variables
    x = [];
    u = [];
    a = [];
    p = [];
    d = [];

	%% Get names of states from obj
    [stateNames, auxNames, ctrlNames, ...
    paramNames, inputNames] = getFieldNames(obj);
    
   
        
    for n=1:length(paramNames)
        p.(paramNames{n}) = obj.p.(paramNames{n}).val;
    end

    for n=1:length(stateNames)
        if isscalar(obj.x.(stateNames{n}).val) % only initial value defined
            x.(stateNames{n}) = obj.x.(stateNames{n}).val; 
        else % trajectory of state defined
            x.(stateNames{n}) = obj.x.(stateNames{n}).val(1,2); 
        end
    end

    
    for n=1:length(inputNames)
        d.(inputNames{n}) = obj.d.(inputNames{n}).val(1,2); % inital input value
    end
    
    for n=1:length(ctrlNames)
        if isscalar(obj.u.(ctrlNames{n}).val) % only initial value defined
            u.(ctrlNames{n}) = obj.u.(ctrlNames{n}).val; 
        elseif isempty(obj.u.(ctrlNames{n}).val)
           try
                u.(ctrlNames{n}) = eval(preEval(obj.u.(ctrlNames{n}).def));
            catch err
                msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement u.%s (n=%d): \n\t''%s''',...
                    err.message, ctrlNames{n}, n, obj.u.(ctrlNames{n}).def);
                id = 'MATLAB:StateSpaceModel:eval';
                error(id,msg);
            end
        else % trajectory of control defined
            u.(ctrlNames{n}) = obj.u.(ctrlNames{n}).val(1,2); 
                % get first value
        end
    end
    
    for n=1:length(auxNames)
        try
            a.(auxNames{n}) = eval(obj.a.(auxNames{n}).def);
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement a.%s (n=%d): \n\t''%s''',...
                err.message, auxNames{n}, n, obj.a.(auxNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end  
    end
    
end

function u1 = getControls(obj, x, a, u0, d, p)
% calculate u1 as a function of x0, d0, a0, u0
% u1 = f(x0,d0,a0,u0)

    [~, ~, ctrlNames, ~, ~] = getFieldNames(obj);
    
    % calculate next step for controls 
    u = u0;
    for n=1:length(ctrlNames)
        try
            u1.(ctrlNames{n}) = eval(preEval(obj.u.(ctrlNames{n}).def)); 
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement u.%s (n=%d): \n\t''%s''',...
                err.message, ctrlNames{n}, n, obj.u.(ctrlNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end
    end
end

function [x1, a1]  = getNextStep(obj, stepSize, x0, a0, u0, uHalf, u1, d0, dHalf, d1, p)
    % calculate values of x1, a1, based on values of x0, a0, u0, u1, d0, d1
    % and odes given in obj.x.<>.def
    %
    % x1 = x0+stepSize*f(x0,d0,u0,a0)
    % a1 = f(x1,d1,u1)
    
    [stateNames, auxNames, ~, ~, ~] = getFieldNames(obj);
    
    % Calculate k1 = stepSize*f(t0,x0)
    k1  = getK(obj, stepSize, x0, a0, u0, d0, p);

    % calculate xHalf = x+k1/2
    for k=1:length(stateNames)
        xHalf.(stateNames{k}) = x0.(stateNames{k})+1/2*k1.(stateNames{k});            
    end

    % calculate aHalf = a(xHalf)
    x = xHalf;
    d = dHalf;
    u = uHalf;
    a = a0;
    for k=1:length(auxNames) 
        try
            aHalf.(auxNames{k}) = eval(obj.a.(auxNames{k}).def);
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement a.%s (n=%d): \n\t''%s''',...
                err.message, auxNames{n}, n, obj.a.(auxNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end
    end

    % Calculate k2 = stepSize*f(tHalf,x0+k1/2)
    k2  = getK(obj, stepSize, xHalf, aHalf, uHalf, dHalf, p);

    % calculate xHalf = x+k2/2
    for k=1:length(stateNames)
        xHalf.(stateNames{k}) = x0.(stateNames{k})+1/2*k2.(stateNames{k});            
    end

    % calculate aHalf = a(xHalf)
    x = xHalf;
    d = dHalf;
    u = uHalf;
    for k=1:length(auxNames)
        try
            aHalf.(auxNames{k}) = eval(obj.a.(auxNames{k}).def);
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement a.%s (n=%d): \n\t''%s''',...
                err.message, auxNames{n}, n, obj.a.(auxNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end
    end

    % Calculate k3 = stepSize*f(tHalf,x0+k2/2)
    k3  = getK(obj, stepSize, xHalf, aHalf, uHalf, dHalf, p);

    % calculate xHalf = x+k3
    for k=1:length(stateNames)
        xHalf.(stateNames{k}) = x0.(stateNames{k})+k3.(stateNames{k});            
    end

    % calculate aHalf = a(xHalf)
    x = xHalf;
    d = dHalf;
    u = uHalf;
    for k=1:length(auxNames)
        try
            aHalf.(auxNames{k}) = eval(obj.a.(auxNames{k}).def);
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement a.%s (n=%d): \n\t''%s''',...
                err.message, auxNames{n}, n, obj.a.(auxNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end
    end

    % Calculate k4 = stepSize*f(t1,x0+k3)
    k4  = getK(obj, stepSize, xHalf, aHalf, u1, d1, p);

    % calculate x1 = x0+1/6(k1+2*k2+2*k3+k4)
    for k=1:length(stateNames)
        x1.(stateNames{k}) = x0.(stateNames{k})+...
            1/6*(k1.(stateNames{k})+2*k2.(stateNames{k})+2*k3.(stateNames{k})...
            +k4.(stateNames{k})); 
    end

    % calculate a1 = a(x1)
    x = x1;
    d = d1;
    u = u1;
    for k=1:length(auxNames)
        try
            a1.(auxNames{k}) = eval(obj.a.(auxNames{k}).def); 
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement a.%s (n=%d): \n\t''%s''',...
                err.message, auxNames{n}, n, obj.a.(auxNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end
    end
end

function k = getK(obj, stepSize, x0, a0, u0, d0, p)
    % get k according to the Runge-Kutta method
    % k = stepSize*f(t,x)
    
    [stateNames, ~, ~, ~, ~] = getFieldNames(obj);
    
    % values to use in calculation of x1
    x = x0;
    u = u0;
    d = d0;
    a = a0;
    for n=1:length(stateNames)
		try
            k.(stateNames{n}) = stepSize*eval(obj.x.(stateNames{n}).def);
        catch err
            msg = sprintf('%s \n\nFailed to evaluate the definition of DynamicElement x.%s (n=%d): \n\t''%s''',...
                err.message, stateNames{n}, n, obj.x.(stateNames{n}).def);
            id = 'MATLAB:StateSpaceModel:eval';
            error(id,msg);
        end
    end
end
